import React, { useEffect, useState } from "react";
import { api } from "../config/api";
import { Link, useNavigate } from "react-router-dom";
import { Plus, FileText, Loader, ArrowRight } from "lucide-react";

export const FormList = () => {
  const [forms, setForms] = useState([]);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    const fetchForms = async () => {
      try {
        const res = await api.get("/api/forms");
        setForms(res.data);
      } catch (err) {
        console.error("Failed to fetch forms", err);
      } finally {
        setLoading(false);
      }
    };
    fetchForms();
  }, []);

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <Loader className="animate-spin text-blue-600" size={32} />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 p-8 font-sans text-neutral-900">
      <div className="max-w-6xl mx-auto">
        <div className="flex justify-between items-center mb-12">
          <h1 className="text-3xl font-light text-neutral-800">Your Forms</h1>
          <Link
            to="/builder"
            className="bg-blue-600 text-white px-6 py-3 rounded-lg flex items-center gap-2 hover:bg-blue-700 transition-colors shadow-lg shadow-blue-100 font-medium"
          >
            <Plus size={20} />
            Create New Form
          </Link>
        </div>

        {forms.length === 0 ? (
          <div className="text-center py-20 bg-white rounded-xl shadow-sm border border-gray-100">
            <FileText size={48} className="mx-auto text-gray-300 mb-4" />
            <h3 className="text-xl font-medium text-gray-800 mb-2">
              No forms yet
            </h3>
            <p className="text-gray-500 mb-8">
              Create your first form to get started gathering responses.
            </p>
            <Link
              to="/builder"
              className="text-blue-600 font-medium hover:text-blue-800 transition-colors"
            >
              Create a form &rarr;
            </Link>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {forms.map((form) => (
              <div
                key={form._id}
                className="bg-white p-6 rounded-xl shadow-sm border border-gray-100 hover:shadow-md transition-shadow group flex flex-col justify-between"
              >
                <div>
                  <div className="flex items-start justify-between mb-4">
                    <div className="p-3 bg-blue-50 rounded-lg text-blue-600">
                      <FileText size={24} />
                    </div>
                    {/* Potential place for actions like delete/edit in future */}
                  </div>
                  <h2 className="text-xl font-medium text-gray-800 mb-2 line-clamp-1">
                    {form.title || "Untitled Form"}
                  </h2>
                  <p className="text-gray-500 text-sm mb-6 line-clamp-2">
                    {form.steps
                      ? `${form.steps.length} questions`
                      : "No questions"}
                  </p>
                </div>

                <div className="flex gap-2 mt-4">
                  <Link
                    to={`/form/${form._id}`}
                    className="flex-1 py-2.5 px-4 rounded-lg border border-gray-200 text-gray-600 font-medium text-center hover:bg-gray-50 hover:text-blue-600 hover:border-blue-200 transition-all flex items-center justify-center gap-2"
                  >
                    View
                    <ArrowRight size={16} />
                  </Link>
                  <Link
                    to={`/form/${form._id}/responses`}
                    className="flex-1 py-2.5 px-4 rounded-lg bg-blue-50 text-blue-600 font-medium text-center hover:bg-blue-100 transition-colors flex items-center justify-center gap-2"
                  >
                    Responses
                  </Link>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
};
